unit tree_unit;

{  DavData software
   www.davdata.nl
   jan. 2013 

   supply tree operations on elements with undo

   maxelement = number of elements
   maxaction  = number of undo steps saved

   element is record with mandatory fields:
   - parent
   - child
   - next
   - previous
   - elType   (elFree = element not in use)
   - + customfields like x,y position, width, height, font, code.......

   operation       description
   ---------       --------------
   placeA          place after
   placeB          place before
   placeChildA     place as child after
   placeChildB     place as child before
   delete          de-activate
   replace         replace

   navigating
   ----------
   parentEL        get parent of element
   ChilddEL        get child of element
   NextEL          get next of element
   PreviousEL      get previous of element
   scanElement     supply neighbour element in tree:
                   old relation      new direction search in priority order
                   -------------     --------------------------------------
                   next,child        child,next,previous,parent
                   previous          previous, parent
                   parent            next,previous,parent

   misc
   ----
   getELstatus     return status of element : stActive,stDelete,stFree
   getFreeEL       return free element to use

   undo
   ----
   undoElement     undo last operation
   purgeUndoStack  remove undostack, destroy deleted elements
   startlinking    next action is linked
   stoplinking     next action is not linked

   global constants
   ----------------
   maxelement      total number of elements
   maxUndo         total number of undo steps remembered

   global variables
   ----------------
   element[1..maxelement]   array of all elements
   freeElementCount         number of free elements available
   undolist[1..maxUndo]     array of undo steps
   freeUndoCount            number of unused Undo steps
   lastAction               highest entry in Undolist
}

interface

uses windows;

type TLinkType      = (ltNone,ltParent,ltChild,ltNext,ltPrevious);
     TElementStatus = (esFree,esActive,esDeleted);
     TELaction    = (eaPlaceA,eaPlaceB,eaPlaceChildA,eaPlaceChildB,eaReplace,
                     eaDelete,eaUndo,eaNone);
     TElementtype = (elFree,elActive);
     TElement = record
                 eltype : TElementType;
                 marked : boolean;
                 x,y    : word;
                 parent : dword;
                 child  : dword;
                 next   : dword;
                 previous : dword
                end;
      TEditAction = record
                     action  : TELaction;
                     elmnt   : dword;     //element
                     linkEL  : dword;     //old element link
                     linkType: Tlinktype;   //type of link
                     bwlink  : boolean;   //backward link
                    end;

const maxelement = 50;
      maxundo    = 25;  //undo actions

var element : array[1..maxelement] of TElement;
    undolist : array[1..maxundo] of TEditAction;
    lastAction : word;
    freeElementcount : dword = 0;
    freeUndocount : word = 0;

//-- initialize
procedure initTreedata;
//--edit procedures
function getFreeEL(var el : dword) : boolean;  //get free element
procedure PlaceELA(del,sel : dword);           //place element sel after del
procedure PlaceELB(del,sel : dword);           //place element sel after del
procedure PlaceChildA(del,sel : dword);        //place elemnt sel after child
procedure PlaceChildB(del,sel : dword);        //place element sel before child
procedure ReplaceEL(del,sel : dword);          //replace element del by sel
procedure deleteEL(del : dword);               //delete element el
//--navigating
function ParentEl(var nel : dword) : boolean;
function ChildEl(var nel : dword) : boolean;
function NextEl(var nel : dword) : boolean;
function previousEl(var nel : dword) : boolean;
function ScanElement(var elmnt : dword; lt : TLinkType) : TLinktype;
//--status
function getELStatus(el : dword) : TElementStatus;
//-- undo stack
procedure startLinking;
procedure stoplinking;
procedure purgeUndoStack;                      //clear all undo information
procedure UndoElement;                         //undo last action

implementation

var freeElementNr : dword;
    startflag,linkflag : boolean;

procedure destroyEL(del : dword);forward;

//---low level support

procedure clearElement(el : dword);
begin
 with element[el] do
  begin
   parent := 0;
   child := 0;
   next := 0;
   previous := 0;
   elType := elFree;
   marked := false;
   x := 0;
   y := 0;
  end;
end;

procedure initUndolist;
var i : word;
begin
 for i := 1 to maxUndo do
  with undolist[i] do
   begin
    elmnt := 0;
    action := eaNone;
    bwlink := false;
   end;
 lastAction := 0;
end;
   
procedure initTreedata;
var i : dword;
begin
 for i := 1 to maxelement do clearElement(i);
 freeElementNr := 1;
 freeElementcount := maxelement;
 initUndoList;
 with element[1] do
  begin
   x := 1;
   y := 1;
  end;
 startflag := false;
 linkflag := false;
end;

function getELlink(var leltype : TLinktype; el : dword) : dword;
//return element connection of el in leltype
//result = link element   0: no link
//1:test next; 2:test previous; 3: test parent
var elmnt : dword;
label exit;
begin
 leltype := ltNone;
 elmnt := element[el].next;
 if elmnt > 0 then
  begin
   leltype := ltNext;
   goto exit;
  end;
 elmnt := element[el].previous;
 if elmnt > 0 then
  begin
   leltype := ltPrevious;
   goto exit;
  end;
 elmnt := element[el].parent;
 if elmnt > 0 then leltype := ltParent;
exit: 
 if leltype <> ltNone then result := elmnt else result := 0;
end;

function pickEL(pel : dword) : TEditAction;
//extract element from tree
//save link info
var par,nxt,prv : dword;
begin
 with result do
  begin
   linkEL := getELlink(linktype,pel);
   if linkType <> ltNone then
    begin
     with element[pel] do
      begin
       par := parent;
       nxt := next;
       prv := previous;
      end;
     if prv > 0 then element[prv].next := nxt
      else if par > 0 then element[par].child := nxt;
     if nxt > 0 then element[nxt].previous := prv;
    end;//if link
  end;//with result
end;

//--- navigating

function getELStatus(el : dword) : TElementStatus;
//esFree   : no parent or previous
//esActive : connected with parent or previous
//esDeleted: not connected to parent or previous
//element 1 always active
begin
 if el = 1 then
  begin
   result := esActive;
   exit;
  end;

 with element[el] do
  begin
   if (parent = 0) and (previous = 0) then
    begin
     result := esFree;
     exit;
    end;

   if parent > 0 then
    if element[parent].child = el then
     begin
      result := esActive;
      exit;
     end;

   if previous > 0 then
    if element[previous].next = el then
     begin
      result:= esActive;
      exit;
     end;
  end;
  result := esDeleted;
end;

function ParentEl(var nel : dword) : boolean;
var e : dword;
begin
 e := element[nel].parent;
 if e > 0 then
  begin
   nel := e;
   result := true;
  end else result := false;
end;

function ChildEl(var nel : dword) : boolean;
var e : dword;
begin
 e := element[nel].child;
 if e > 0 then
  begin
   nel := e;
   result := true;
  end else result := false;
end;

function NextEl(var nel : dword) : boolean;
var e : dword;
begin
 e := element[nel].next;
 if e > 0 then
  begin
   nel := e;
   result := true;
  end else result := false;
end;

function previousEl(var nel : dword) : boolean;
var e : dword;
begin
 e := element[nel].previous;
 if e > 0 then
  begin
   nel := e;
   result := true;
  end else result := false;
end;

//--- undo ---

procedure startLinking;
//start new edit commands
//clear linkflag, set startflag
begin
 linkflag := false;
 startFlag := true;
end;

procedure stopLinking;
begin
 startflag := false;
 linkflag := false;
end;

procedure dropUndo(nr : word);
//erase undo action nr
begin
 with undolist[nr] do
  if (action = eaReplace) or (action = eaDelete) then destroyEL(elmnt);
end;

procedure shiftUndoStack;
//remove bottom of undo stack to make space
var link : boolean;
    i : word;
begin
 if lastAction = 0 then exit;
 repeat
  dropUndo(1);
  if lastaction > 1 then link := undolist[2].bwlink
   else link := false;
  for i := 1 to lastAction - 1 do undolist[i] := undolist[i+1];
  with undolist[lastaction] do
   begin
    elmnt := 0;
    linkEL := 0;
    linkType := ltNone;
    action := eaNone;
    bwlink := false;
   end;
  dec(lastaction);
 until link = false;
end;

procedure registerAction(act : TEditAction);
//add link flag to action, register in undo list
begin
 if lastAction = maxUndo then shiftUndoStack;
 inc(lastAction);
 act.bwlink := linkflag;
 undolist[lastaction] := act;
 linkflag := startflag;
end;

procedure UndoElement;
var par,prv,nxt : dword;
    bwl : boolean;
begin
 repeat
  with Undolist[lastaction] do
   begin
    bwl := bwLink;
    if action <> eaDelete then pickEL(elmnt);
    case action of
     eaPlaceA,
     eaPlaceB,
     eaPlaceChildA,
     eaPlaceChildB,
     eaDelete      : begin
                      with element[linkEL] do
                       begin
                        par := parent;
                        nxt := next;
                        prv := previous;
                       end;
                      case linkType of
                       ltParent   : begin
                                     element[linkEL].child := elmnt;
                                     with element[elmnt] do
                                      begin
                                       parent := linkEL;
                                       next := 0;
                                       previous := 0;
                                      end;
                                    end;
                       ltNext     : begin
                                     element[linkEL].previous := elmnt;
                                     with element[elmnt] do
                                      begin
                                       parent := par;
                                       next := linkEL;
                                       previous := prv;
                                      end;
                                     if prv = 0 then
                                      element[par].child := elmnt
                                     else element[prv].next := elmnt;
                                     if nxt > 0 then
                                      element[linkEL].previous := elmnt;
                                    end;
                       ltPrevious : begin
                                     element[linkEL].next := elmnt;
                                      with element[elmnt] do
                                       begin
                                        parent := par;
                                        previous := linkEL;
                                        next := 0;
                                       end;
                                    end;
                       ltNone     : destroyEL(elmnt);
                      end;//case linktype
                     end;//eaDelete
     eaReplace     : begin
                      with element[linkEL] do
                       begin
                        prv := previous;
                        nxt := next;
                        par := parent;
                       end;
                      if prv > 0 then element[prv].next := elmnt
                       else element[par].child := elmnt;
                      if nxt > 0 then element[nxt].previous := elmnt;
                      with element[linkEL] do
                       begin                //extract from tree
                        parent := 0;
                        next := 0;
                        previous := 0;
                       end;
                      destroyEL(linkEL);
                     end;
    end;//case action
   end;//with undolist
  dec(lastaction);
 until bwl = false;
end;

procedure purgeUndoStack;
var i : word;
begin
 for i := 1 to lastAction do dropUndo(i);
 initUndoList;
end;

//--- element actions ---

procedure PlaceELA(del,sel : dword);
//place element sel after del
var nxt,par : DWORD;
    ea      : TEditAction;
begin
 ea := pickEL(sel);      //free from tree
 ea.action := eaPlaceA;
 ea.elmnt := sel;
 with element[del] do
  begin
   nxt := next;
   par := parent;
  end;
 element[del].next := sel;
 with element[sel] do
  begin
   previous := del;
   parent := par;
   next := nxt;
  end;
 if nxt > 0 then element[nxt].previous := sel;
 if ea.linkType = ltNone then dec(freeElementCount);
 registerAction(ea);
end;

procedure PlaceELB(del,sel : dword);
//insert new element before e
var prv,par : DWORD;
    ea : TEditAction;
begin
 ea := pickEL(sel);      //free from tree
 ea.action := eaPlaceB;
 ea.elmnt := sel;
 with element[del] do
  begin
   prv := previous;
   par := parent;
  end;
 element[del].previous := sel;
 with element[sel] do
  begin
   previous := prv;
   if previous = 0 then element[par].child := sel
    else element[prv].next := sel;
   next := del;
   parent := par;
  end;
 if ea.linkType = ltNone then dec(freeElementCount);
 registerAction(ea);
end;

procedure PlaceChildA(del,sel : dword);
var ea  : TEditAction;
    chd : dword;
begin
 ea := pickEL(sel);      //free from tree
 ea.action := eaPlaceChildB;
 ea.elmnt := sel;
 chd := element[del].child;
 repeat until nextEL(chd) = false; //position on last child
 with element[sel] do
  begin
   parent := del;
   next := 0;
   previous := chd;
  end;
 if chd = 0 then element[del].child := sel
  else element[chd].next := sel;
 if ea.linkType = ltNone then dec(freeElementCount);
 registeraction(ea);
end;

procedure PlaceChildB(del,sel : dword);
var ea  : TEditAction;
    chd : dword;
begin
 ea := pickEL(sel);      //free from tree
 ea.action := eaPlaceChildA;
 ea.elmnt := sel;
 chd := element[del].child;
 with element[sel] do
  begin
   parent := del;
   previous := 0;
   next := chd;
  end;
 element[del].child := sel;
 if chd > 0 then element[chd].previous := sel;
 if ea.linkType = ltNone then dec(freeElementCount);
 registerAction(ea);
end;

procedure ReplaceEL(del,sel : dword);
//replace element del by sel
//sel must be free element
var nxt,prv,par : dword;
    ea : TeditAction;
begin
 with ea do
  begin
   elmnt := del;            //replaced element
   linkEL := sel;           //replacing element
   linktype := ltNone;
   action := eaReplace;
  end;
 with element[del] do
  begin
   par := parent;
   nxt := next;
   prv := previous;
  end;
 with element[sel] do
  begin
   parent := par;
   previous := prv;
   next := nxt;
  end;
 if prv > 0 then element[prv].next := sel
  else element[par].child := sel;
 if nxt > 0 then element[nxt].previous := sel;
 dec(freeElementCount);
 registerAction(ea);
end;

procedure deleteEL(del : dword);
//delete element del
//do not delete children
var ea : TEditAction;
begin
 ea := pickEL(del);
 ea.action := eaDelete;
 ea.elmnt := del;
 registerAction(ea);
end;

procedure destroyEL(del : dword);
//destroy element and it's children
//adjust freeElementcount
//element may not be part of tree
var mcode : TLinktype;
    oldEl : dword;
begin
 with element[del] do
  begin
   if getELStatus(del) = esActive then PickEL(del); //delete first if active
   parent := 0;       //isolate element, keep link to children
   next := 0;
   previous := 0;
  end;
 mcode := ltChild;
 repeat
  OldEl := del;
  mcode := ScanElement(del,mcode);
  if (mcode = ltPrevious) or (mcode = ltParent) or (mcode = ltNone) then
   begin
    clearElement(oldEl);
    inc(freeElementcount);
   end;
 until mcode = ltNone;
end;

function ScanElement(var elmnt : dword; lt : TLinktype) : TLinktype;
//walk thru tree, children first then next (down)
//replace elmnt by next element
begin
 result := ltNone;
 if (lt = ltChild) or (lt = ltNext) then
  begin
   if childEl(elmnt) then result := ltChild
    else if nextEl(elmnt) then result := ltNext
          else if previousEl(elmnt) then result := ltPrevious
           else if parentEl(elmnt) then result := ltParent;
  end;
 if result <> ltNone then exit;

 if lt = ltPrevious then
  begin
   if previousEl(elmnt) then result := ltPrevious
    else if parentEl(elmnt) then result := ltParent;
  end;
 if result <> ltNone then exit;

 if lt = ltParent then
  begin
   if nextEl(elmnt) then result := ltNext
    else if previousEl(elmnt) then result := ltPrevious
     else if parentEl(elmnt) then result:= ltParent;
  end;
end;

function getFreeEL(var el : dword) : boolean;
//return free element number el
//return true if Ok, false if out of elements
//decrement freeElementcount
var n   : DWORD;
begin
 n := freeElementNr;
 repeat
  result := element[n].elType = elFree;
  if result then el := n
   else begin
         inc(n);
         if n > maxelement then n := 1;
        end;
 until result or (n = freeElementNr);
 if result then
  begin
   freeElementNr := n+1;
   if freeElementNr > maxElement then freeElementNr := 1;
   dec(freeElementcount);
   with element[n] do    //reset some element props
    begin
     child := 0;
     next := 0;
    end;
  end;
end;

end.
