unit poly_unit;

{  version 1.0

   polygon procedures and functions

   main calls:

   - function  polyArea(pts:pointer; n:byte) : double;   //area of polygon

   - procedure setCanvas(cvs : Tcanvas);                 //select canvas

   - procedure setpencolor(pc : longInt);                //pen color
   - procedure setBGcolor(bgc : longInt);                //background color
   - procedure setBG(BGmode : boolean);                  //BG color true/false

   - procedure geoClear;                                 //debug purposes
   - procedure setShowTriangles(trMode : boolean);       //triangle mode on/off

   - function polyResultMessage : string;                //message of code

   output

   polyresultcode  = 0 : OK
                     1 : insufficient points
                     2 : polygon not closed
                     3 : cannot triangulate
                     4 : duplicate points
                     5 : intersecting edge
}

interface

uses types,graphics;

const maxpolypoint = 100;
      polysquare = 400; //20 * 20 pixels

var polyresultcode : byte;
    polyTime : longInt;

procedure setcanvas(cvs : Tcanvas);//select canvas for draw/fill of polygon
procedure setpencolor(pc: longInt);
procedure setBGcolor(bgc : longInt);
procedure setBGmode(bgM : boolean);
procedure setShowTriangles(trMode : boolean);
function polyArea(pts : pointer; n : byte) : double; //area of polygon
function getPolyResultMessage : string;

implementation

uses clock_unit;

const pi05 = 0.5*pi;
      pi15 = 1.5*pi;
      pi2 = 2*pi;
      rd05 = 0.5;                  //rounding
      frnd = 1e-6;                 //floating point round to zero

type Tpoints = array[1..maxpolypoint] of Tpoint;
     Ppoints = ^Tpoints;

     Tvector = record
                x,y,dx,dy : smallInt;
                dir : double;       //direction 0..2*pi
               end;

     TVectorList = array[1..maxpolypoint] of TVector;

     TTriangle = record
                  ax,ay,bx,by,cx,cy : smallInt; //coordinates of angles A,B,C
                 end;

     TTriangleList = array[1..maxpolypoint] of TTriangle;

var pcount : byte;
    pp : Ppoints;
    vectorlist : TVectorlist;
    vcount : byte;
    triangleList : Ttrianglelist;
    tcount : byte;
    fvalid : boolean = false;
    f1,f2 : double;
    pts : ^TVectorlist;
    Area : double;

    cv : Tcanvas = nil;
    pencolor : LongInt = $000000;
    BGcolor : longInt = $a0a0a0;
    BGmode : boolean = false;
    trianglemode : boolean;

procedure checkpoints;forward;
procedure buildVectorlist;forward;
procedure checkvectors;forward;
procedure buildtrianglelist;forward;
procedure TriangleSums;forward;
procedure drawpoly;forward;

procedure geoClear;
//reset all data , debug purposes only
var i : word;
begin
 for i := 1 to maxpolypoint do
  with Vectorlist[i] do
   begin
    x := 0;
    y := 0;
    dx := 0;
    dy := 0;
    dir := 0;
   end;
 Vcount := 0;
 for i := 1 to maxpolypoint do
  with TriangleList[i] do
   begin
    ax := 0; ay := 0;
    bx := 0; by := 0;
    cx := 0; cy := 0;
   end;
 tCount := 0;
end;

procedure setcanvas(cvs : Tcanvas);
begin
 cv := cvs;
end;

procedure setpencolor(pc : longInt);
//set color of the pen
begin
 pencolor := pc;
end;

procedure setBGcolor(bgc : longInt);
//set background color
begin
 BGcolor := bgc;
 BGmode := true;
end;

procedure setBGmode(BGm : boolean);
//switch BG color on/off
begin
 BGmode := BGm;
end;

procedure setShowTriangles(trMode : boolean);
//set triangle mode on/off for debug purposes
//show triangles if set true
begin
 trianglemode := trMode;
end;

procedure polyDraw(pts:pointer; n:byte);
//draw polygon on previously selected canvas
begin
 polyresultcode := 0;
 pp := Ppoints(pts);
 pcount := n;
 checkpoints;
 if polyresultcode <> 0 then exit;
 buildvectorlist;
 checkvectors;
 if polyresultcode <> 0 then exit;

 buildtrianglelist;
 if polyresultcode <> 0 then exit;

end;

function  polyArea(pts:pointer; n:byte) : double;
//calculate area, draw polygon
var t1,t2 : Int64;
begin
 getCPUticks(t1);
 polyresultcode := 0;
 pp := Ppoints(pts);
 pcount := n;
 checkpoints;
 if polyresultcode <> 0 then exit;

 buildvectorlist;
 checkvectors;
 if polyresultcode <> 0 then exit;

 buildtrianglelist;
 if polyresultcode <> 0 then exit;

 trianglesums;
 result := area/polysquare;
 getCPUticks(t2);
 polyTime := ProcTime(t2-t1);
 drawpoly;
end;

// ---- low level support procedures ----------

function VDir(deltaX,deltaY : double) : double;
//return direction of vector in radians
//(+,0) = 0; (0,+) = 0.5pi ; (-,0) = pi ; (0,-) = 1.5pi
begin
 if deltaX = 0 then
  begin
   if deltaY > 0 then result := pi05 else result := pi15;
   exit;
  end;
 result := arctan((deltaY)/(deltaX));
 if deltaX < 0 then result := result + pi;
 if result < 0 then result := result + pi2;
end;

procedure vrsect(const v1,v2 : TVector);
//calculate intersection of vectors v1,v2
//line1 = (v1.x1,v1.y1) +f1*(v1.dx,v1.dy)
//line2 = (v2.x1,v2.y1) +f2*(v2.dx,v2.dy)
//return f1,f2,fvalid
var d,vx,vy : double;
begin
 d := v1.dx*v2.dy - v1.dy*v2.dx;//discriminant
 if d = 0 then begin
                fvalid := false; exit;
               end;
 fvalid := true;
 vx := v2.x - v1.x;
 vy := v2.y - v1.y;
 f1 := (vx*v2.dy - vy*v2.dx)/d;
 f2 := (vx*v1.dy - vy*v1.dx)/d;

 if abs(f1) < frnd then f1 := 0;  //round to 1e-6
 if abs(f2) < frnd then f2 := 0;
 if abs(f1-1) < frnd then f1 := 1;
 if abs(f2-1) < frnd then f2 := 1;
end;

function Adjacent(a,b : word): boolean;
//called by "checkvectors"  (geoVectorcount must be set before)
//return true if vectors Vlist[a],Vlist[b] share start- endpoint
begin
 result := ((a = 1) and (b = vcount)) or
           ((b = 1) and (a = vcount)) or
           (abs(a-b) < 2);
end;

function outward(vnr : word) : boolean;
//return true if  vlist[vnr] points outward
var i : word;
    crosscount : shortInt;
    VAngle : double;
begin
 crosscount := 0;
 for i := 1 to Vcount do
  begin
   vrsect(vectorlist[vnr],vectorlist[i]);
     if fvalid and (f1 > 1) then
      begin
       VAngle := vectorlist[i].dir - vectorlist[vnr].dir;
       if Vangle < 0 then Vangle := Vangle + pi2;
       if Vangle < pi then
        begin
         if (f2 = 0) or (f2 = 1) then inc(crosscount);   //left crossing
         if (f2 > 0) and (f2 < 1) then inc(crosscount,2);//..
        end;
       if VAngle > pi then
        begin
         if (f2 = 0) or (f2 = 1) then dec(crosscount);   //right crossing
         if (f2 > 0) and (f2 < 1) then dec(crosscount,2);//..
        end;
    end;//if fvalid..
  end;//for
 result := crosscount = 0;
end;

function Empty3(i1,i2,i3 : word) : boolean;
//check for no point inside triangle i1,i2 (i3)
//no point : true
var k  : word;
    sv,sw : TVector;
begin
 with sv do
  begin
   x := vectorlist[i1].x;
   y := vectorlist[i1].y;
   dx := vectorlist[i3].x - vectorlist[i1].x;
   dy := vectorlist[i3].y - vectorlist[i1].y;
  end;
 for k := 1 to vcount do
  if (k <> i1) and (k <> i2) and (k <> i3) then
   begin
    with sw do
     begin
      x := vectorlist[i2].x;
      y := vectorlist[i2].y;
      dx := vectorlist[k].x - x;
      dy := vectorlist[k].y - y;
     end;
    vrsect(sv,sw);
    if fvalid and (f2 >= 1) and (f1 > 0) and (f1 < 1) then 
     begin
      result := false;
      exit;
     end;//if fvalid
  end;//for
 result := true;
end;

procedure geoTriColor(const tr:Ttriangle;brushcol:LongInt);
//color triangle with brushCol
//v1 must point to start of v2
var h : integer;
    a : array[1..3] of TPoint;
    x1,y1,x2,y2,x3,y3 : integer;
    dx1,dx2,dx3,x,xx : single;
    ii,jj : byte;
begin
 dx1 := 1; dx2 := 1; dx3 := 1;
 with tr do
  begin
   a[1].x := ax;                    //load array
   a[1].y := ay;
   a[2].x := bx;
   a[2].y := by;
   a[3].x := cx;
   a[3].y := cy;
  end;
 for ii := 1 to 2 do                          //sort array y - incr.
  for jj := ii+1 to 3 do
   if a[ii].y > a[jj].y then
    begin
     h := a[ii].x; a[ii].x := a[jj].x; a[jj].x := h;
     h := a[ii].y; a[ii].y := a[jj].y; a[jj].y := h;
    end;
 x1 := a[1].x; y1 := a[1].y;                  //load sorted points
 x2 := a[2].x; y2 := a[2].y;
 x3 := a[3].x; y3 := a[3].y;
 if y1 <> y3 then dx3 := (x3-x1)/(y3-y1);
 if y1 <> y2 then
  begin
   dx1 := (x2-x1)/(y2-y1);
   xx := x1;
  end
 else xx := x2;
 if y2 <> y3 then dx2 := (x3-x2)/(y3-y2);
 x := x1;
 with cv do
  begin
   pen.Width := 1;
   pen.color := BGcolor;
   for h := y1 to y3 do
    begin
     moveto(trunc(x+0.5),h);
     lineto(trunc(xx+0.5),h);
     x := x + dx3;
     if h >= y2 then xx := xx + dx2 else xx := xx + dx1;
   end;
  end; 
end;

function getPolyresultmessage : string;
//set message string according to polyresultcode
begin
 case polyresultcode of
  0 : result := 'OK';
  1 : result := 'insufficient points';
  2 : result := 'polygon not closed';
  3 : result := 'cannot triangulate';
  4 : result := 'duplicate point';
  5 : result := 'intersecting edge';
 end;//case
end;

//---- hi level support procedures --------------

//--- build Vectorlist

procedure buildVectorlist;
var n : word;
begin
 for n := 1 to pcount-1 do
  with vectorlist[n] do
  begin
   x := pp^[n].x;
   y := pp^[n].Y;
   dx := pp^[n+1].X - x;
   dy := pp^[n+1].Y - y;
   dir := VDir(dx,dy);   //direction of vector
  end;//with
 vcount := pcount-1; 
end;

//--- check points ----

procedure checkpoints;
//set polyresultcode if not closed polygon or insufficient points
begin
 if pcount < 4 then
  begin
   polyresultcode := 1;
   exit;
  end;

 if (pp^[1].x <> pp^[pcount].X) or (pp^[1].y <> pp^[pcount].y) then
  begin
   polyresultcode := 2;
   exit;
  end;
end;

//--- check vectorlist

procedure checkvectors;
//check Vlist for
// - duplicate points
// - intersections
var i,j : word;
begin
 for i := 1 to pcount-1 do                    //check duplicate points
 for j := i+1 to pcount do
  if (vectorlist[i].x = vectorlist[j].x) and
     (vectorlist[i].y = vectorlist[j].y) then
   begin
    polyResultCode := 4;             //duplicate point
    exit;
   end;

 for i := 1 to pcount do        //check for no intersection
  for j := 1 to pcount do
   if adjacent(i,j) = false then
    begin
     vrsect(vectorlist[i],vectorlist[j]);
     if fvalid and ((f1 >= 0) and (f1 <= 1)) and ((f2 >= 0) and (f2 <= 1))then
      begin
       polyResultcode := 5;  //intersection of edges
       exit;
      end;
    end;
end;

procedure buildTriangleList;
var i,j,ni,nni : word;
    OK : boolean;

label loop1,loop2,end1;

begin
 tCount := 0;
 i := 1;

loop1:

 if i > 1 then dec(i);

loop2:

 ni := i + 1;
 if ni > Vcount then ni := 1; //correct overflow
 nni := ni + 1;
 if nni > Vcount then nni := 1;
 OK := outward(i);
 if OK then OK := Empty3(i,ni,nni);
 if OK then
  begin
   inc(tCount);
   with triangleList[tCount] do
    begin
     ax := Vectorlist[i].x;
     ay := Vectorlist[i].y;
     bx := Vectorlist[ni].x;
     by := Vectorlist[ni].y;
     cx := Vectorlist[nni].x;
     cy := Vectorlist[nni].y;
    end;
   with vectorlist[i] do              //replace vlist[i] by vlist[i]+vlist[ni]
    begin
     dx := dx + vectorlist[ni].dx;
     dy := dy + vectorlist[ni].dy;
     dir := Vdir(dx,dy);
    end;
   for j := ni to vcount-1 do
       vectorlist[j] := vectorlist[j+1];//eliminate vlist[ni]
   dec(vcount);
   if vcount >= 3 then goto loop1 else
    begin
     polyResultCode := 0;  //OK
     exit;
    end;
  end
 else         //not OK
  begin
   inc(i);
   if i <= vcount then goto loop2;
   polyResultCode := 3;           //triangulation error
  end;
end;

procedure TriangleSums;
//use triangles list to calculate area
var i : word;
    a,b,c,s,dx,dy : double;
begin
 Area := 0;
 for i := 1 to tcount do
  with triangleList[i] do
   begin
    dx := ax - bx;
    dy := ay - by;
    a := sqrt(dx * dx + dy * dy);
    dx := bx - cx;
    dy := by - cy;
    b := sqrt(dx * dx + dy * dy);
    dx := cx - ax;
    dy := cy - ay;
    c := sqrt(dx * dx + dy * dy);
    s := 0.5*(a + b + c);
    Area := Area + sqrt(abs(s*(s-a)*(s-b)*(s-c)));//abs for spurious -0.0000001
   end;//for..with
end;

procedure drawpoint(p : Tpoint);
//draw point on canvas cv
var r : Trect;
begin
  with r do
  begin
   Left := p.x - 2;
   Right := p.x + 3;
   Top := p.y - 2;
   Bottom := p.y + 3;
  end;
 with cv do
  begin
   brush.color := $000000;
   pen.color := $000000;
   fillrect(r);
  end;
end;

procedure drawline(p1,p2 : Tpoint);
//draw line from p1 to p2 on canvas cv
begin
 with cv do
  begin
   moveto(p1.X,p1.Y);
   lineto(p2.X,p2.Y);
  end;
end;

procedure drawpoly;
//draw polygon in cabvas cv
var i : byte;
begin
 with cv do
  begin
   if BGmode then
    begin
     pen.Width := 1;
     for i := 1 to tcount do geotricolor(trianglelist[i],BGcolor);
    end;
   if triangleMode then
    begin
     pen.Color := $0000ff;
     for i := 1 to tCount do
      with trianglelist[i] do
       begin
        moveto(ax,ay);
        lineto(bx,by);
        lineto(cx,cy);
        lineto(ax,ay);
       end;
    end;
   pen.color := pencolor;
   pen.Width := 2;

   for i := 1 to pcount-1 do
    begin
     drawline(pp^[i],pp^[i+1]);
     drawpoint(pp^[i]);
    end;
  end;//with CV
end;

end.
