unit compressunit;

{ PIC : pixel image compression

  DavData Software
  4 - 05 - 2014
  www.davdata.nl

  12 bit colors
  -------------

  00000000 RRRR1111 GGGG1111 BBBB01111   32 bit format          (dword)

  0000 rrrr gggg bbbb                    internal 12 bit format (word)

   12 bit color directors
   ----------------------

  00 00  rrrr gggg bbbb  new color
  00 01                  Hmarker
  00 10                  Vmarker
  00 11  rgb ddd         delta
  01     sssss           color from stack sssss [0..31]
  10                     copy left pixel
  11                     copy top  pixel

  4 bit color greyscale directors
  -------------------------------

  00 00    delta +
  00 01    HMarker
  00 10    VMarker
  00 11    delta -
  01 gggg  new color
  10       copy left
  11       copy top

  version 3
  - removed delta colors
  - stack 64 entries
  - rounded 12 bit colors to $1f,$2f....$ff (1,2,3.....$f are 4 bit colors)
                                            ($0x unused)

  director list format
  --------------------

           C          I          P         S
  rrrr  tttt  hhhh hhhh hhhh  wwww wwww wwww        rrrr:revision tttt:type
  ..............#3.........#2.............#1        directors
                                       .....        ....

  version 4
  - address mem by pointer (was [x,y])
  - scanning in 4*4 pixel groups in 12 bit mode (more stack hits)
  - set border (-1) color to $ff
  - changed name to 'pic'

  version 5
  - changed instruction codes
  - reduced stack from 64 to 32 entries
  - changed name to "spic"
  - changed file format
  - changed block size to 16 * 16
  - added "delta" rgb ddd instruction  r,g,b +-1 of +-2 from Left color
  - 4 bit Hor. & Vert. expansion codes
  - changed grey code to add Vmarker
  - added delta +/- to grey codes
  - changed grey codes for same marker codes grey/color
  
  bitstream compression
  ---------------------

  If LL... detected then marker set to mkLeft, Hmarker code inserted.
  If TT... detected then marker set to mkTop , Vmarker code inserted.
  A LLLL... in mkLeft mode writes -1-
  A TTTT... in mktop mode write -1-
  When no copy : -0- bit is written to terminate compression mode

  version 5
  - enlarged max picture dimensions to 4000 * 3000  (hor * vert)
  - added scrollbars
  - added saving of .bmp files
  - added loading of .jpg images
}

interface

uses windows,graphics,dialogs,sysutils;

type Tstatistics = (stLeft,stExpLeft,stTop,stExpTop,stStack,stDelta,stNew);

var statistics : array[stLeft..stNew] of dword;
    dircount  : dword;  //length of director table
    colorcount : dword = 0;

procedure setMap(mp : Tbitmap);
procedure clearAll;
procedure ResetDir;
procedure colorfix12;
procedure greyScale;
procedure compressgrey;
procedure compress12;
procedure decompress(var cc : byte);
function getdirector(n : dword) : dword;
function storedirector(dw : dword) : dword;
procedure scancolors;
procedure decompressgrey;
procedure decompress12;

implementation

type PDW = ^dword;
     TMarker = (mkNone,mkLeft,mkTop);

const maxdir      = 5000000;  //directory space
      stacklength = 32;
      blocksize   = 16;       //dimensions of scanblock
      spic        = 'spic';   //file identifier
      
//code generation

      HMcount = 4;   //horizontal marker count
      HCcount = 4;   //   ...     copy   ...
      VMcount = 4;   //vertical marker count
      VCcount = 4;   //    ...  copy   ...

var director    : array[0..maxdir] of dword;
    dirPtr      : dword;
    accu        : dword;  //bit asembly register
    acount      : byte;   //nr of valid bitsin accu
    map         : Tbitmap;
    p0          : dword;  //left top canvas pointer
    ps          : dword;  //line downstep value
    cc          : array[0..127] of dword; //colorcount bits
    stack       : array[0..stacklength-1] of word;   //rrrrggggbbbb

// compression / expansion

    Hcount      : byte;
    Vcount      : byte;
    expansioncount : byte;
    marker      : TMarker;

//-- low level helpers --

procedure resetDir;
//call before decompression
var i : byte;
begin
 accu := 0;
 acount := 0;
 dirPtr := 0;
 for i := 0 to stacklength-1 do stack[i] := $fff;
 marker := mkNone;
 Hcount := 0;
 Vcount := 0;
 expansioncount := 0;
end;

procedure clearAll;
//call before compression
var n : dword;
    s : Tstatistics;
begin
 for n := 0 to maxDir do director[n] := 0;
 dircount := 0;
 for s := stLeft to stNew do statistics[s] := 0;
 resetDir;
end;

//--- color counting

procedure clearCC;
//clear color count array
var i : word;
begin
 for i := 0 to 127 do cc[i] := 0;
end;

procedure setcolorbit(r,g,b : byte);
//set color bit
const mask = $f0;
var col,row,k : word;
begin
 r := r and mask;
 g := g and mask;
 b := b and mask;
 k := (b shr 4) or g or (r shl 4);
 col := k and $1f;
 row := k shr 5;
 cc[row] := cc[row] or (1 shl col);
end;

procedure countcolors;
var i,j : word;
begin
 colorcount := 0;
 for j := 0 to 127 do
  for i := 0 to 31 do colorcount := colorcount + ((cc[j] shr i) and 1);
end;

function XYtoP(x,y : word) : dword;
//make pointer to map in memory for [x,y] pixel
begin
 result := p0 - y*ps + (x shl 2);
end;

//-- color handling

function getcolor12(p : dword) : word;
//get color in rrrgggbbb format
var color : dword;
begin
 color := PDW(p)^;
 result := ((color shr 4) and $f) or
           ((color shr 8) and $f0) or
           ((color shr 12) and $f00);
end;

function getcolor4(p : dword) : byte;
//get [x,y] map color from blue
begin
 result := (PDW(p)^ shr 4) and $f;
end;

function fixRGB12(c : byte) : byte;
//round 4 bit to $1f,$2f.....$ff
var c1 :  word;
begin
 c1 := (c + $8) and $1f0;
 if c1 < $20 then c1 := $20;
 dec(c1);
 result := c1 and $ff;
end;

function color32to12(d : dword) : word;
//convert 00000000rrrr000gggg0000bbbb0000 to 0000rrrrggggbbbb
begin
 result := ((d shr 4) and $f) or ((d shr 8) and $f0) or ((d shr 12) and $f00);
end;

function color12to32(w : word) : dword;
//convert 0000 rrrr gggg bbbb to 00000000 rrrr1111 gggg1111 bbbb1111
begin
 result := ((w shl 4) and $f0) or
           ((w shl 8) and $f000) or
           ((w shl 12) and $f00000) or $000f0f0f;
end;

function color4to32(c : byte) : dword;
//convert cccc to 00000000 cccc1111 cccc1111 cccc1111
begin
 result := (c shl 4) or (c shl 12) or (c shl 20) or $f0f0f; 
end;

procedure colorfix12;
//reduce colors in map to 12 bit
var x,y : word;
    Pline,p  : dword;
    color : dword;
    r,g,b : word;
begin
 clearCC;  //colorcount array
 for y := 0 to map.Height-1 do
  begin
   Pline := p0 - y*ps;
   for x := 0 to map.Width-1 do
    begin
     p := pline + (x shl 2);
     color := PDW(p)^;
     r := (color shr 16) and $ff;
     g := (color shr 8) and $ff;
     b := color and $ff;
     r := fixRGB12(r);
     g := fixRGB12(g);
     b := fixRGB12(b);
     color := b or (g shl 8) or (r shl 16);
     PDW(p)^ := color;
     setcolorbit(r,g,b);         //for colorcount;
    end;
  end;
 countcolors;
end;

//--- stack

function checkstack(c : word; var stk : byte) : boolean;
//true if color c is in stack
//stk is index (0..63)
begin
 stk := stacklength;
 repeat
  dec(stk);
  result := stack[stk] = c;
 until result or (stk = 0);
end;

procedure enterstack(c : word);
//enter color c in stack
//stack 63 is top, 0 is bottom
var i,stk : byte;
begin
 checkstack(c,stk);
 for i := stk to stacklength-2 do
  begin
   stack[i] := stack[i+1];
  end;
 stack[stacklength-1] := c;
end;

function checkdelta(new,old : word; var code : byte) : boolean;
//check for +-1 of rgb colors
//code: color $1      0:+1  1:+2
//            $2..$e  0:+1  1:-1
//            $f      0:-2  1:-1
var a : array[0..2] of byte;
    b : array[0..2] of byte;
    i,rgb,delta,mask : byte;
begin
 result := true;
 for i := 0 to 2 do
  begin
   a[i] := new shr (i shl 2) and $f;
   b[i] := old shr (i shl 2) and $f;
  end;
 rgb := 0;
 delta := 0;
 for i := 0 to 2 do
  begin
   mask := 1 shl i;
   case a[i] - b[i] of
    -2 : if b[i] = $f then rgb := rgb or mask else result := false;
    -1 : begin
          rgb := rgb or mask;
          delta := delta or mask;
         end;
     0 : ;
     1 : rgb := rgb or mask;
     2 : begin
          if b[i] = $1 then rgb := rgb or mask else result := false;
          delta := delta or mask;
         end;
     else result := false;
    end;
  end;//for i
 code := (rgb shl 3) or delta;
end;

function Deltacolor(color : word; code : byte) : word;
//make new color code from old color and code rgbddd
var a : array[0..2] of byte;
    i,dif : byte;
begin
 for i := 0 to 2 do a[i] := (color shr (i shl 2)) and $f;
 for i := 0 to 2 do
  if ((1 shl (i+3)) and code) <> 0 then   //if enabled
   begin
    if (code shr i) and 1 = 0 then
     begin
      if (a[i] = $f) then dif := $e else dif := 1;
     end
     else if (a[i] = $1) then dif := 2 else dif := $f;
    a[i] := (a[i] + dif) and $f;
   end;
 result := 0;
 for i := 0 to 2 do result := result or (a[i] shl (i shl 2));
end;

function checkdeltaGrey(new,old : byte; var cc : byte) : boolean;
//cc=0(+) of 3(-)
begin
 cc := 0;
 result := false;
 case new - old of
  -2 : if old = $f then result := true;
  -1 : begin
        cc := 3;
        result := true;
       end;
   1 : result := true;
   2 : if old = $1 then
        begin
         result := true;
         cc := 3;
        end;
 end;//case
end;

function deltaGrey(col,code : byte) : byte;
//code 0..1
begin
 if code = 0 then
  begin
   if col = $f then result := col - 2 else result := col + 1;
  end
  else if col = $1 then result := col + 2 else result := col - 1;
end;

//--- bit stream assembly / disassembly

procedure storeCode(w : dword; n : byte);
//store n bits of w to dirtab (via accu)
//n = 1 ..16
var space : byte;
begin
 space := 32 - acount;
 accu := accu or (w shl acount);
 if space >= n then            //if space
  begin
   acount := acount + n;
   if acount = 32 then
    begin
     acount := 0;
     director[dirPtr] := accu;
     inc(dirPtr);
     accu := 0;
    end;
  end
  else                         //if no space
   begin
    director[dirPtr] := accu;
    accu := 0;
    inc(dirPtr);
    accu := w shr space;
    acount := n - space;
   end;
end;

function readcode(n : byte) : dword;
//read n bits from director table
var mask : dword;
    space : byte;
begin
 if acount= 0 then
  begin
   accu := director[dirPtr];
   inc(dirPtr);
  end;
 mask := (1 shl n) -1;
 space := 32 - acount;
 result := accu shr acount;
 acount := acount + n;
 if acount = 32 then acount := 0
  else if acount > 32 then
        begin
         accu := director[dirPtr];
         inc(dirPtr);
         dec(acount,32);
         result := (result or (accu shl space));
        end;
 result := result and mask;
end;

//------- main calls ------------------

procedure setMap(mp : Tbitmap);
begin
 map := mp;
 with map do
  begin
   PixelFormat := pf32bit;
   p0 := dword(ScanLine[0]);        //p0 in compressunit
   ps := p0 - dword(scanline[1]);   //ps................
  end;
end;

procedure greyScale;
//set map to grey
var pline,p,color : dword;
    x,y,w : word;
    r,g,b : byte;
begin
 if not assigned(map) then exit;

 clearCC;
 for y := 0 to map.height-1 do
  begin
   pline := p0 - ps*y;
   for x := 0 to map.Width-1 do
    begin
     p := pline + (x shl 2);
     color := PDW(p)^;
     r := (color shr 16) and $ff;
     g := (color shr 8) and $ff;
     b := color and $ff;
     w := (r + g + b + 2) div 3;
     r := (w and $f0) or $f;
     setcolorbit(r,r,r);
     color := (r shl 16) or (r shl 8) or r;
     PDW(p)^ := color;
    end;//for x
  end;//for y
 countcolors;
end;

//-- compression helpers

procedure cleanUpH;
var k : byte;
begin
 if Hcount > 0 then
  begin
   for k := 1 to Hcount do storecode(2,2);
   inc(statistics[stLeft],Hcount);
   Hcount := 0;
  end;
end;

procedure cleanUpV;
var k : byte;
begin
 if Vcount > 0 then
  begin
   for k := 1 to Vcount do storecode(3,2);
   inc(statistics[stTop],Vcount);
   Vcount := 0;
  end;
end;

//--- compress  / decompress

procedure compressGrey;
//director pointers must be cleared
var x,y : word;
    cLeft,cNew,cTop : byte;
    p : dword;
    cc : byte;  //free variables
begin
 for x := 1 to 4 do storecode(byte(spic[x]),8);  //file type
 storecode(map.Width,12);
 storecode(map.Height,12);
 storecode(4,4);                                //compression mode
 storecode(1,4);                                //revision #
 for y := 0 to map.Height-1 do
  for x := 0 to map.Width-1 do
   begin
    p := p0 - y*ps + (x shl 2);
    if x > 0 then cLeft := getcolor4(p-4) else cLeft := $f;
    if y > 0 then cTop := getColor4(p+ps) else cTop := $f;
    cNew := getcolor4(p);
    cc := 0;                           //copy code 1:left  2:top
    if cNew = cLeft then cc := 1;
    if cNew = cTop then cc := cc or 2;
    if cc = 3 then
     if Vcount > 0 then cc := 2 else cc := 1;

     case cc of      //1:left
      $1  : case marker of
             mkNone : begin            //LC & no marker
                       cleanUpV;
                       inc(Hcount);
                       if Hcount = HMcount then
                        begin
                         Hcount := 0;
                         marker := mkLeft;
                         storecode(0,2);                 //store Hmarker 00 01
                         storecode(1,2);
                         inc(statistics[stExpLeft],HMcount); //Hmarker count
                        end;
                      end;
             mkLeft : begin            //LC & HMarker
                       inc(Hcount);
                       if Hcount = HCcount then
                        begin
                         storecode(1,1);
                         inc(statistics[stExpLeft],HCcount); //Hcopy count
                         Hcount := 0;
                        end;
                      end;
              mkTop : begin           //LC & VMarker
                       storecode(0,1);
                       marker := mkNone;
                       cleanUpV;
                       Hcount := 1;
                      end;
            end; //case
         $2  : case marker of    //2:Vmark
          mkNone  : begin           //TC & no Marker
                     cleanUpH;
                     inc(Vcount);
                     if Vcount = VMcount then
                      begin
                       VCount := 0;
                       marker := mkTop;
                       storecode(0,2);                    //store Vmarker 00 10
                       storecode(2,2);
                       inc(statistics[stExpTop],VMcount);
                      end;
                    end;
          mkLeft  : begin           //TC & HMarker
                     storecode(0,1);
                     marker := mkNone;
                     cleanUpH;
                     Vcount := 1;
                    end;
            mkTop : begin           //TC & VMarker
                     inc(Vcount);
                     if Vcount = VCcount then
                      begin
                       storecode(1,1);
                       inc(statistics[stExpTop],VCcount);
                       Vcount := 0;
                      end;
                    end;
               end;
         $0 : begin         //no copy
               if marker <> mkNone then storecode(0,1); //unmark
               marker := mkNone;
               cleanUpH;
               cleanUpV;

               if checkdeltaGrey(cNew,cLeft,cc) then  //check delta
                begin
                 storecode(0,2);
                 storecode(cc,2);         //cc: 0+  3-
                 inc(statistics[stdelta]);
                end
               else
               begin
                storecode(1,2);          //new code
                storecode(cNew,4);       //color
                inc(statistics[stNew]);
               end;
              end;
        end; //case

    end;//for x,y

 if marker <> mkNone then storecode(0,1);
 cleanUpH;
 cleanUpV;
 if acount > 0 then storecode(0,32-acount);//store any unsaved code
 dircount := dirPtr;
end;

procedure decompressGrey;
//called from common decompression proc.
//file type verified, map dimensions stripped allready
//decompress compression file type 4
var x,y : word;
    cLeft,Ctop : byte;
    code : byte;
    p : dword;
    color : dword;
    color4 : byte;
begin
 color4 := $f;
 for y := 0 to map.Height-1 do
  for x := 0 to map.Width-1 do
   begin
    p := XYtoP(x,y);
    if x=0 then cLeft :=$f else cLeft := getcolor4(p-4);
    if y=0 then cTop :=$f else cTop :=getcolor4(p+ps);
    if marker <> mkNone then code := 4 else code := readcode(2);
//
    case code of
     1 : color4 := readcode(4);  //new code
     0 : begin
          code := readcode(2);   //get subcode
          case code of
           0,
           3 : color4 := deltaGrey(cLeft,code);
           1 : begin
                marker := mkLeft;
                color4 := cLeft;
                expansioncount := HMcount - 1;
               end;
           2 : begin
                marker := mkTop;
                color4 := cTop;
                expansioncount := VMcount - 1;
               end;
          end;//case
         end;//1
     2 : color4 := cLeft;
     3 : color4 := cTop;
     4 : begin                 //if expansion active
          dec(expansioncount);
          if marker = mkLeft then
           begin
            color4 := cLeft;
            if expansioncount = 0 then
             if readcode(1) = 1 then expansioncount := HCcount
              else marker := mkNone;
           end;
          if marker = mkTop then
           begin
            color4 := cTop;
            if expansioncount = 0 then
             if readcode(1) = 1 then expansioncount := VCcount
              else marker := mkNone;
           end;
         end;//4
     end;//case
     color := color4to32(color4);
     PDW(p)^ := color;
  end;//for x,y
end;

procedure compress12;
//director pointers must be cleared
//scanning in blocksize * blocksize , left top to right bottom
var gx,gy,x,y : word;
    cLeft,cNew,cTop : word;
    i,j,w,h : byte;
    color,p : dword;
    stk,cc : byte;  //free variables
begin
 for x := 1 to 4 do storecode(byte(spic[x]),8);  //file type
 storecode(map.Width,12);
 storecode(map.Height,12);
 storecode(12,4);                                //compression mode
 storecode(1,4);                                 //revision #
 for gy := 0 to ((map.Height-1) div blocksize) do
  begin
   y := gy * blocksize;
   if y + blocksize > map.Height then h := map.Height-y else h := blocksize;
   for gx := 0 to ((map.Width-1) div blocksize)  do
    begin
     x := gx * blocksize;
     if x + blocksize > map.width then w := map.width-x else w := blocksize;
     for j := 0 to h-1 do
      for i := 0 to w-1 do
       begin
        p := XYtoP(x+i,y+j);
        if x+i > 0 then cLeft := getcolor12(p-4) else cLeft := $fff;
        if y+j > 0 then cTop := getColor12(p+ps) else cTop := $fff;
        color := PDW(p)^;
        cNew := color32to12(color);
        cc := 0;                           //copy code 1:left  2:top
        if cNew = cLeft then cc := 1;
        if cNew = cTop then cc := cc or 2;

        if cc = 3 then
         if Vcount > 0 then cc := 2 else cc := 1;

        case cc of      //1:left
         $1  : case marker of
           mkNone : begin            //LC & no marker
                     cleanUpV;
                     inc(Hcount);
                     if Hcount = HMcount then
                      begin
                       Hcount := 0;
                       marker := mkLeft;
                       storecode(0,2);                   //store Hmarker 00 01
                       storecode(1,2);
                       inc(statistics[stExpLeft],HMcount); //Hmarker count
                      end;
                    end;
           mkLeft : begin            //LC & HMarker
                     inc(Hcount);
                     if Hcount = HCcount then
                      begin
                       storecode(1,1);
                       inc(statistics[stExpLeft],HCcount); //Hcopy count
                       Hcount := 0;
                      end;
                    end;
            mkTop : begin           //LC & VMarker
                     storecode(0,1);
                     marker := mkNone;
                     cleanUpV;
                     Hcount := 1;
                    end;
               end;
         $2  : case marker of    //2:Vmark
          mkNone  : begin           //TC & no Marker
                     cleanUpH;
                     inc(Vcount);
                     if Vcount = VMcount then
                      begin
                       VCount := 0;
                       marker := mkTop;
                       storecode(0,2);                    //store Vmarker 00 10
                       storecode(2,2);
                       inc(statistics[stExpTop],VMcount);
                      end;
                    end;
          mkLeft  : begin           //TC & HMarker
                     storecode(0,1);
                     marker := mkNone;
                     cleanUpH;
                     Vcount := 1;
                    end;
            mkTop : begin           //TC & VMarker
                     inc(Vcount);
                     if Vcount = VCcount then
                      begin
                       storecode(1,1);
                       inc(statistics[stExpTop],VCcount);
                       Vcount := 0;
                      end;
                    end;
               end;
         $0 : begin         //no copy
               if marker <> mkNone then storecode(0,1); //unmark
               marker := mkNone;
               cleanUpH;
               cleanUpV;
               if checkstack(cNew,stk) then      //check stack
                begin
                 storecode(1,2);
                 storecode(stk,5);               //0..32
                 inc(statistics[stStack]);
                end
               else

               if checkdelta(cNew,cLeft,cc) then  //check delta
                begin
                 storecode(0,2);
                 storecode(3,2);
                 storecode(cc,6);
                 inc(statistics[stDelta]);
                end
               else
               begin
                storecode(0,4);           //new code
                storecode(cNew,12);       //color
                inc(statistics[stNew]);
               end;
              end;
        end; //case

        enterstack(cNew);
       end;//for i,j
    end;//for gx
  end;//for gy

 if marker <> mkNone then storecode(0,1);
 cleanUpH;
 cleanUpV;
 if acount > 0 then storecode(0,32-acount);//store any unsaved code
 dircount := dirPtr;
end;

procedure decompress12;
//called from common decompression proc.
//file verified, map dimensions stripped allready
//decompress compression file type 12
var gx,gy,x,y : word;
    stk,code,i,j,w,h : byte;
    color,p : dword;
    color12,cLeft,cTop : word;
begin
 color12 := $fff;
 for gy := 0 to ((map.Height-1) div blocksize) do
  begin
   y := gy * blocksize;
   if y + blocksize > map.Height then h := map.Height-y else h := blocksize;
   for gx := 0 to ((map.Width-1) div blocksize)  do
    begin
     x := gx * blocksize;
     if x + blocksize > map.width then w := map.width-x else w := blocksize;
     for j := 0 to h-1 do
      for i := 0 to w-1 do
       begin
        p := XYtoP(x+i,y+j);
        if marker <> mkNone then code := 4 else code := readcode(2);
        if x+i=0 then cLeft := $fff else cLeft := getcolor12(p-4);
        if y+j=0 then cTop := $fff else cTop := getcolor12(p+ps);
//
        case code of
         0 : begin
              code := readcode(2);    //get subcode
              case code of
               0 : color12 := readcode(12);
               1 : begin
                    color12 := cLeft;
                    marker := mkLeft;
                    expansioncount := HMcount - 1;
                   end;
               2 : begin
                    color12 := cTop;
                    marker := mkTop;
                    expansioncount := VMcount - 1;
                   end;
               3 : begin
                    color12 := cLeft;
                    code := readcode(6);
                    color12 := DeltaColor(color12,code);
                   end;
              end;
             end;
         1 : begin                    //from stack
              stk := readcode(5);
              color12 := stack[stk];
             end;
         2 : begin                    //copy left
              color12 := cLeft;
             end;
         3 : begin                    //copy top
              color12 := cTop;
             end;
         4 : begin                 //if expansion active
              dec(expansioncount);
              if marker = mkLeft then
               begin
                color12 := cLeft;
                if expansioncount = 0 then
                 if readcode(1) = 1 then expansioncount := HCcount
                  else marker := mkNone;
               end;
              if marker = mkTop then
               begin
                color12 := cTop;
                if expansioncount = 0 then
                 if readcode(1) = 1 then expansioncount := VCcount
                  else marker := mkNone;
               end;
             end;//4
        end;//case
        color := color12to32(color12);
        enterstack(color12);
        PDW(p)^ := color;
       end;//for i,j
    end;//for gx
  end;//for gy
end;

// --- main call

procedure decompress(var cc : byte);
//decompress director table , 4 or 12 bits
//map is destination bitmap, size is adjusted
//exit cc=0 if OK
var i : byte;
    s : string;
    grey : boolean;
begin
 s := '';
 for i := 1 to 4 do s := s + char(readcode(8));
 if s <> 'spic' then
  begin
   cc := 1;                      //wrong file type
   exit;
  end;

 with map do
  begin
   width := readcode(12);
   height := readcode(12);
   p0 := dword(scanline[0]);
   ps := p0 - dword(scanline[1]);
  end;

 case readcode(4) of
  4   : grey := true;
  12  : grey := false;
  else begin
        cc := 2;                 //wrong format
        exit;
       end;
 end;//case

 readcode(4);                    //skip revision#
 cc := 0;
 if grey then decompressGrey else decompress12;
end;

function getdirector(n : dword) : dword;
//for testing
begin
 result := director[n];
end;

function storedirector(dw : dword) : dword;
begin
 director[dirPtr] := dw;
 inc(dirPtr);
 result := dirPtr;
end;

procedure scancolors;
//scan map & count colors
var x,y : word;
    pline,p,dw : dword;
    r,g,b : byte;
begin
 clearCC;
 for y := 0 to map.Height-1 do
  begin
   pline := p0 - y*ps;
   for x := 0 to map.width-1 do
    begin
     p := pline + (x shl 2);
     dw := PDW(p)^;
     r := (dw shr 16) and $f0;
     g := (dw shr 8) and $f0;
     b := dw and $f0;
     setcolorbit(r,g,b);
    end;
  end;
 countcolors;
end;

end.
